<?php

class Database
{
    private $host = 'localhost';
    private $db_name = 'qztdvwbd_bot';
    private $username = 'qztdvwbd_user';
    private $password = 'TopManager@90';
    private $conn;

    public function __construct()
    {
        $this->getConnection();
    }

    // Get database connection
    public function getConnection()
    {
        if ($this->conn !== null) {
            return $this->conn;
        }

        try {
            $this->conn = new PDO(
                "mysql:host=" . $this->host . ";dbname=" . $this->db_name,
                $this->username,
                $this->password
            );
            $this->conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->conn->exec("set names utf8");
            return $this->conn;
        } catch (PDOException $e) {
            error_log("Connection Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Create necessary tables if they don't exist
    public function initializeTables()
    {
        try {
            $queries = [
                // Users table with more details
                "CREATE TABLE IF NOT EXISTS users (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    chat_id BIGINT NOT NULL UNIQUE,
                    phone_number VARCHAR(15),
                    first_name VARCHAR(50),
                    last_name VARCHAR(50),
                    credit_score INT DEFAULT 0,
                    loan_limit DECIMAL(10,2) DEFAULT 0.00,
                    status VARCHAR(20) DEFAULT 'active',
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
                )",

                // Transactions table
                "CREATE TABLE IF NOT EXISTS transactions (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    user_id INT,
                    service_type VARCHAR(50),
                    amount DECIMAL(10,2),
                    status VARCHAR(20),
                    mpesa_receipt VARCHAR(50),
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    FOREIGN KEY (user_id) REFERENCES users(id)
                )",

                // Service requests table
                "CREATE TABLE IF NOT EXISTS service_requests (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    user_id INT,
                    service_type VARCHAR(50),
                    status VARCHAR(20),
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    FOREIGN KEY (user_id) REFERENCES users(id)
                )",

                // Loans table with more details
                "CREATE TABLE IF NOT EXISTS loans (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    user_id INT,
                    amount DECIMAL(10,2),
                    interest_rate DECIMAL(5,2) DEFAULT 15.00,
                    due_date DATE,
                    status VARCHAR(20) DEFAULT 'pending',
                    mpesa_receipt VARCHAR(50),
                    disbursement_date TIMESTAMP NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    FOREIGN KEY (user_id) REFERENCES users(id)
                )",

                // Loan transactions table with more details
                "CREATE TABLE IF NOT EXISTS loan_transactions (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    merchant_request_id VARCHAR(50),
                    user_id INT,
                    loan_id INT NULL,
                    amount DECIMAL(10,2),
                    transaction_type ENUM('processing_fee', 'disbursement', 'repayment') DEFAULT 'processing_fee',
                    status VARCHAR(20),
                    mpesa_receipt VARCHAR(50),
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    FOREIGN KEY (user_id) REFERENCES users(id),
                    FOREIGN KEY (loan_id) REFERENCES loans(id)
                )"
            ];

            foreach ($queries as $query) {
                $this->conn->exec($query);
            }

            return true;
        } catch (PDOException $e) {
            error_log("Table Creation Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Helper function to create or update user
    public function saveUser($chat_id, $phone_number = null, $first_name = null, $last_name = null)
    {
        try {
            $query = "INSERT INTO users (chat_id, phone_number, first_name, last_name) 
                     VALUES (:chat_id, :phone_number, :first_name, :last_name) 
                     ON DUPLICATE KEY UPDATE 
                     phone_number = COALESCE(:phone_number, phone_number),
                     first_name = COALESCE(:first_name, first_name),
                     last_name = COALESCE(:last_name, last_name)";

            $stmt = $this->conn->prepare($query);
            $stmt->execute([
                ':chat_id' => $chat_id,
                ':phone_number' => $phone_number,
                ':first_name' => $first_name,
                ':last_name' => $last_name
            ]);

            return $this->conn->lastInsertId() ?: $this->getUserId($chat_id);
        } catch (PDOException $e) {
            error_log("User Save Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Update user's credit score and loan limit
    public function updateUserCreditInfo($user_id, $credit_score, $loan_limit)
    {
        try {
            $query = "UPDATE users 
                     SET credit_score = :credit_score, 
                         loan_limit = :loan_limit 
                     WHERE id = :user_id";

            $stmt = $this->conn->prepare($query);
            return $stmt->execute([
                ':user_id' => $user_id,
                ':credit_score' => $credit_score,
                ':loan_limit' => $loan_limit
            ]);
        } catch (PDOException $e) {
            error_log("Update User Credit Info Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Get user ID by chat_id
    public function getUserId($chat_id)
    {
        try {
            $query = "SELECT id FROM users WHERE chat_id = :chat_id";
            $stmt = $this->conn->prepare($query);
            $stmt->execute([':chat_id' => $chat_id]);

            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            error_log("Get User Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Get user by phone number
    public function getUserByPhone($phone)
    {
        try {
            $query = "SELECT * FROM users WHERE phone_number = :phone";
            $stmt = $this->conn->prepare($query);
            $stmt->execute([':phone' => $phone]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get User Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Create new transaction
    public function createTransaction($user_id, $service_type, $amount)
    {
        try {
            $query = "INSERT INTO transactions (user_id, service_type, amount, status) 
                     VALUES (:user_id, :service_type, :amount, 'pending')";

            $stmt = $this->conn->prepare($query);
            $stmt->execute([
                ':user_id' => $user_id,
                ':service_type' => $service_type,
                ':amount' => $amount
            ]);

            return $this->conn->lastInsertId();
        } catch (PDOException $e) {
            error_log("Transaction Creation Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Create a new loan
    public function createLoan($user_id, $amount, $due_date, $mpesa_receipt)
    {
        try {
            // First check if a loan with this mpesa_receipt already exists
            $query = "SELECT id FROM loans WHERE mpesa_receipt = :mpesa_receipt";
            $stmt = $this->conn->prepare($query);
            $stmt->execute([':mpesa_receipt' => $mpesa_receipt]);
            $existing_loan = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($existing_loan) {
                error_log("Loan already exists with mpesa_receipt: " . $mpesa_receipt);
                return $existing_loan['id'];
            }

            // Create new loan with completed status since payment is confirmed
            $query = "INSERT INTO loans (user_id, amount, due_date, mpesa_receipt, status, disbursement_date) 
                     VALUES (:user_id, :amount, :due_date, :mpesa_receipt, 'completed', NOW())";

            $stmt = $this->conn->prepare($query);
            $result = $stmt->execute([
                ':user_id' => $user_id,
                ':amount' => $amount,
                ':due_date' => $due_date,
                ':mpesa_receipt' => $mpesa_receipt
            ]);

            if (!$result) {
                error_log("Failed to create loan: " . json_encode($stmt->errorInfo()));
                throw new Exception("Failed to create loan");
            }

            $loan_id = $this->conn->lastInsertId();
            error_log("Created loan with ID: " . $loan_id . ", Amount: " . $amount . ", Receipt: " . $mpesa_receipt);
            return $loan_id;
        } catch (PDOException $e) {
            error_log("Create Loan Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Create a new loan transaction
    public function createLoanTransaction($merchant_request_id, $user_id, $amount)
    {
        try {
            $query = "INSERT INTO loan_transactions (merchant_request_id, user_id, amount, status) 
                     VALUES (:merchant_request_id, :user_id, :amount, 'pending')";

            $stmt = $this->conn->prepare($query);
            $stmt->execute([
                ':merchant_request_id' => $merchant_request_id,
                ':user_id' => $user_id,
                ':amount' => $amount
            ]);

            return $this->conn->lastInsertId();
        } catch (PDOException $e) {
            error_log("Create Loan Transaction Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Update loan transaction with loan_id
    public function updateLoanTransactionLoanId($merchant_request_id, $loan_id)
    {
        try {
            $query = "UPDATE loan_transactions 
                     SET loan_id = :loan_id,
                         status = 'completed',
                         updated_at = NOW()
                     WHERE merchant_request_id = :merchant_request_id";

            $stmt = $this->conn->prepare($query);
            $result = $stmt->execute([
                ':loan_id' => $loan_id,
                ':merchant_request_id' => $merchant_request_id
            ]);

            if (!$result) {
                error_log("Failed to update loan transaction: " . json_encode($stmt->errorInfo()));
                throw new Exception("Failed to update loan transaction");
            }

            error_log("Updated loan transaction merchant_request_id: " . $merchant_request_id . " with loan_id: " . $loan_id);
            return true;
        } catch (PDOException $e) {
            error_log("Update Loan Transaction Loan ID Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Update transaction status
    public function updateTransactionStatus($transaction_id, $status, $mpesa_receipt = null)
    {
        try {
            $query = "UPDATE transactions 
                     SET status = :status, mpesa_receipt = :mpesa_receipt 
                     WHERE id = :id";

            $stmt = $this->conn->prepare($query);
            return $stmt->execute([
                ':id' => $transaction_id,
                ':status' => $status,
                ':mpesa_receipt' => $mpesa_receipt
            ]);
        } catch (PDOException $e) {
            error_log("Transaction Update Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Update loan transaction status with more detailed logging
    public function updateLoanTransaction($merchant_request_id, $status, $mpesa_receipt = null)
    {
        try {
            $query = "UPDATE loan_transactions 
                     SET status = :status, 
                         mpesa_receipt = :mpesa_receipt,
                         updated_at = NOW()
                     WHERE merchant_request_id = :merchant_request_id";

            $stmt = $this->conn->prepare($query);
            $result = $stmt->execute([
                ':status' => $status,
                ':mpesa_receipt' => $mpesa_receipt,
                ':merchant_request_id' => $merchant_request_id
            ]);

            if (!$result) {
                error_log("Failed to update loan transaction status: " . json_encode($stmt->errorInfo()));
                throw new Exception("Failed to update loan transaction status");
            }

            error_log("Updated loan transaction status - ID: " . $merchant_request_id .
                ", Status: " . $status .
                ", Receipt: " . ($mpesa_receipt ?? 'null'));
            return true;
        } catch (PDOException $e) {
            error_log("Update Loan Transaction Error: " . $e->getMessage());
            throw $e;
        }
    }

    // Get loan transaction by merchant request ID
    public function getLoanTransaction($merchant_request_id)
    {
        try {
            $query = "SELECT * FROM loan_transactions WHERE merchant_request_id = :merchant_request_id";
            $stmt = $this->conn->prepare($query);
            $stmt->execute([':merchant_request_id' => $merchant_request_id]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get Loan Transaction Error: " . $e->getMessage());
            throw $e;
        }
    }
}
